import { useState } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { StockItem, DailySale, DailySaleItem } from "@/data/stockData";
import { Plus, Trash2, ShoppingCart } from "lucide-react";

interface AddDailySaleDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  stockItems: StockItem[];
  onSubmit: (sale: DailySale, saleItems: { itemName: string; qty: number }[]) => void;
}

interface SaleLineItem {
  itemName: string;
  qty: number;
  costPrice: number;
  sellingPrice: number;
}

const AddDailySaleDialog = ({
  open,
  onOpenChange,
  stockItems,
  onSubmit,
}: AddDailySaleDialogProps) => {
  const [date, setDate] = useState(new Date().toLocaleDateString("en-GB"));
  const [customerName, setCustomerName] = useState("");
  const [customerPhone, setCustomerPhone] = useState("");
  const [customerAddress, setCustomerAddress] = useState("");
  const [paidAmount, setPaidAmount] = useState("");
  const [discount, setDiscount] = useState("");
  const [lines, setLines] = useState<SaleLineItem[]>([]);
  const [selectedItem, setSelectedItem] = useState("");
  const [qty, setQty] = useState("");

  const availableItems = stockItems.filter((si) => si.stockBalance > 0);

  const addLine = () => {
    if (!selectedItem || !qty || Number(qty) <= 0) return;
    const stock = stockItems.find((si) => si.itemName === selectedItem);
    if (!stock) return;

    const quantity = Math.min(Number(qty), stock.stockBalance);

    // Check if item already added
    const existingIndex = lines.findIndex((l) => l.itemName === selectedItem);
    if (existingIndex !== -1) {
      const updated = [...lines];
      updated[existingIndex].qty += quantity;
      setLines(updated);
    } else {
      setLines([
        ...lines,
        {
          itemName: selectedItem,
          qty: quantity,
          costPrice: stock.costPrice,
          sellingPrice: stock.sellingPrice,
        },
      ]);
    }
    setSelectedItem("");
    setQty("");
  };

  const removeLine = (index: number) => {
    setLines(lines.filter((_, i) => i !== index));
  };

  const totalTP = lines.reduce((s, l) => s + l.qty * l.sellingPrice, 0);
  const totalDP = lines.reduce((s, l) => s + l.qty * l.costPrice, 0);
  const totalProfit = totalTP - totalDP;
  const discountVal = Number(discount) || 0;
  const paidVal = Number(paidAmount) || 0;
  const dueAmount = Math.max(0, totalTP - discountVal - paidVal);
  const netProfit = totalProfit - discountVal;

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (lines.length === 0) return;

    const saleItems: DailySaleItem[] = lines.map((l) => ({
      itemName: l.itemName,
      qty: l.qty,
      dp: l.costPrice,
      tp: l.sellingPrice,
      totalTP: l.qty * l.sellingPrice,
      totalDP: l.qty * l.costPrice,
      profit: l.qty * (l.sellingPrice - l.costPrice),
    }));

    const sale: DailySale = {
      date,
      customerName: customerName.trim() || undefined,
      customerPhone: customerPhone.trim() || undefined,
      customerAddress: customerAddress.trim() || undefined,
      items: saleItems,
      totalTP,
      totalDP,
      totalProfit,
      dueAmount,
      paidAmount: paidVal,
      discount: discountVal,
      netProfit,
    };

    const stockOutItems = lines.map((l) => ({
      itemName: l.itemName,
      qty: l.qty,
    }));

    onSubmit(sale, stockOutItems);
    resetForm();
    onOpenChange(false);
  };

  const resetForm = () => {
    setDate(new Date().toLocaleDateString("en-GB"));
    setCustomerName("");
    setCustomerPhone("");
    setCustomerAddress("");
    setPaidAmount("");
    setDiscount("");
    setLines([]);
    setSelectedItem("");
    setQty("");
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-3xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <ShoppingCart className="h-5 w-5 text-primary" />
            Record Daily Sale
          </DialogTitle>
          <DialogDescription>
            Add sold items. Stock will be automatically deducted from inventory.
          </DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          {/* Date */}
          <div className="space-y-2">
            <Label htmlFor="saleDate">Sale Date</Label>
            <Input
              id="saleDate"
              value={date}
              onChange={(e) => setDate(e.target.value)}
              placeholder="DD/MM/YYYY"
            />
          </div>

          {/* Customer Details */}
          <div className="border border-border rounded-lg p-4 space-y-3">
            <h4 className="text-sm font-medium">Customer Details (Optional)</h4>
            <div className="grid grid-cols-2 gap-3">
              <div className="space-y-2">
                <Label htmlFor="customerName">Customer Name</Label>
                <Input
                  id="customerName"
                  value={customerName}
                  onChange={(e) => setCustomerName(e.target.value)}
                  placeholder="Enter customer name"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="customerPhone">Phone Number</Label>
                <Input
                  id="customerPhone"
                  value={customerPhone}
                  onChange={(e) => setCustomerPhone(e.target.value)}
                  placeholder="01XXXXXXXXX"
                />
              </div>
            </div>
            <div className="space-y-2">
              <Label htmlFor="customerAddress">Address</Label>
              <Textarea
                id="customerAddress"
                value={customerAddress}
                onChange={(e) => setCustomerAddress(e.target.value)}
                placeholder="Enter customer address"
                rows={2}
              />
            </div>
          </div>

          {/* Add item row */}
          <div className="flex flex-col sm:flex-row gap-2 items-end">
            <div className="flex-1 space-y-1">
              <Label>Item</Label>
              <Select value={selectedItem} onValueChange={setSelectedItem}>
                <SelectTrigger>
                  <SelectValue placeholder="Select item..." />
                </SelectTrigger>
                <SelectContent>
                  {availableItems.map((si) => (
                    <SelectItem key={si.itemName} value={si.itemName}>
                      {si.itemName} (stock: {si.stockBalance})
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="w-24 space-y-1">
              <Label>Qty</Label>
              <Input
                type="number"
                min="1"
                value={qty}
                onChange={(e) => setQty(e.target.value)}
                placeholder="0"
              />
            </div>
            <Button type="button" variant="outline" size="icon" onClick={addLine} className="flex-shrink-0">
              <Plus className="h-4 w-4" />
            </Button>
          </div>

          {/* Items table */}
          {lines.length > 0 && (
            <div className="border border-border rounded-lg overflow-hidden">
              <Table>
                <TableHeader>
                  <TableRow className="bg-muted/50">
                    <TableHead className="text-xs">Item</TableHead>
                    <TableHead className="text-right text-xs">Qty</TableHead>
                    <TableHead className="text-right text-xs">Cost</TableHead>
                    <TableHead className="text-right text-xs">Price</TableHead>
                    <TableHead className="text-right text-xs">Total</TableHead>
                    <TableHead className="w-10"></TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {lines.map((line, i) => (
                    <TableRow key={i}>
                      <TableCell className="text-sm">{line.itemName}</TableCell>
                      <TableCell className="text-right text-sm">{line.qty}</TableCell>
                      <TableCell className="text-right text-sm text-muted-foreground">
                        ৳{line.costPrice}
                      </TableCell>
                      <TableCell className="text-right text-sm text-muted-foreground">
                        ৳{line.sellingPrice}
                      </TableCell>
                      <TableCell className="text-right text-sm font-medium text-primary">
                        ৳{(line.qty * line.sellingPrice).toLocaleString()}
                      </TableCell>
                      <TableCell>
                        <Button
                          type="button"
                          variant="ghost"
                          size="icon"
                          className="h-7 w-7 text-destructive hover:text-destructive hover:bg-destructive/10"
                          onClick={() => removeLine(i)}
                        >
                          <Trash2 className="h-3.5 w-3.5" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </div>
          )}

          {/* Payment details */}
          <div className="grid grid-cols-2 gap-3">
            <div className="space-y-2">
              <Label htmlFor="paidAmount">Paid Amount (৳)</Label>
              <Input
                id="paidAmount"
                type="number"
                min="0"
                value={paidAmount}
                onChange={(e) => setPaidAmount(e.target.value)}
                placeholder="0"
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="discount">Discount (৳)</Label>
              <Input
                id="discount"
                type="number"
                min="0"
                value={discount}
                onChange={(e) => setDiscount(e.target.value)}
                placeholder="0"
              />
            </div>
          </div>

          {/* Summary */}
          {lines.length > 0 && (
            <div className="bg-muted/30 rounded-lg p-3 space-y-1 text-sm">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Total Sales</span>
                <span className="font-semibold text-primary">৳{totalTP.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Total Cost</span>
                <span>৳{totalDP.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Profit</span>
                <span className="font-semibold text-success">৳{totalProfit.toLocaleString()}</span>
              </div>
              {discountVal > 0 && (
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Discount</span>
                  <span className="text-destructive">-৳{discountVal.toLocaleString()}</span>
                </div>
              )}
              {dueAmount > 0 && (
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Due</span>
                  <span className="text-destructive">৳{dueAmount.toLocaleString()}</span>
                </div>
              )}
              <div className="flex justify-between border-t border-border pt-1 mt-1">
                <span className="font-semibold">Net Profit</span>
                <span className="font-bold text-foreground">৳{netProfit.toLocaleString()}</span>
              </div>
            </div>
          )}

          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit" disabled={lines.length === 0}>
              Record Sale
            </Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  );
};

export default AddDailySaleDialog;
