import { useState } from "react";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogDescription,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { StockItem } from "@/data/stockData";
import { Plus } from "lucide-react";

interface AddStockDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  onAdd: (item: StockItem) => void;
}

const AddStockDialog = ({ open, onOpenChange, onAdd }: AddStockDialogProps) => {
  const [form, setForm] = useState({
    itemName: "",
    vendor: "",
    stockInQty: "",
    costPrice: "",
    sellingPrice: "",
    stockInDate: new Date().toLocaleDateString("en-GB"),
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (!form.itemName || !form.stockInQty || !form.costPrice || !form.sellingPrice) return;

    const qty = Number(form.stockInQty);
    const cost = Number(form.costPrice);
    const price = Number(form.sellingPrice);

    const newItem: StockItem = {
      itemName: form.itemName,
      vendor: form.vendor || undefined,
      stockInDate: form.stockInDate,
      stockInQty: qty,
      stockOutDate: "",
      stockOutQty: 0,
      costPrice: cost,
      stockBalance: qty,
      sellingPrice: price,
      totalSalesRate: 0,
      totalStockRate: qty * cost,
    };

    onAdd(newItem);
    setForm({ itemName: "", vendor: "", stockInQty: "", costPrice: "", sellingPrice: "", stockInDate: new Date().toLocaleDateString("en-GB") });
    onOpenChange(false);
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="sm:max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Plus className="h-5 w-5 text-primary" />
            Add New Stock Item
          </DialogTitle>
          <DialogDescription>Fill in the details for the new product.</DialogDescription>
        </DialogHeader>
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="itemName">Item Name</Label>
            <Input
              id="itemName"
              value={form.itemName}
              onChange={(e) => setForm({ ...form, itemName: e.target.value })}
              placeholder="e.g. Notebook Large"
              required
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="vendor">Vendor <span className="text-muted-foreground text-xs">(optional)</span></Label>
            <Input
              id="vendor"
              value={form.vendor}
              onChange={(e) => setForm({ ...form, vendor: e.target.value })}
              placeholder="e.g. ABC Suppliers"
            />
          </div>
          <div className="grid grid-cols-2 gap-3">
            <div className="space-y-2">
              <Label htmlFor="stockInQty">Initial Quantity</Label>
              <Input
                id="stockInQty"
                type="number"
                min="1"
                value={form.stockInQty}
                onChange={(e) => setForm({ ...form, stockInQty: e.target.value })}
                placeholder="0"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="stockInDate">Stock In Date</Label>
              <Input
                id="stockInDate"
                value={form.stockInDate}
                onChange={(e) => setForm({ ...form, stockInDate: e.target.value })}
                placeholder="DD/MM/YYYY"
              />
            </div>
          </div>
          <div className="grid grid-cols-2 gap-3">
            <div className="space-y-2">
              <Label htmlFor="costPrice">Cost Price (৳)</Label>
              <Input
                id="costPrice"
                type="number"
                min="0"
                step="0.01"
                value={form.costPrice}
                onChange={(e) => setForm({ ...form, costPrice: e.target.value })}
                placeholder="0"
                required
              />
            </div>
            <div className="space-y-2">
              <Label htmlFor="sellingPrice">Selling Price (৳)</Label>
              <Input
                id="sellingPrice"
                type="number"
                min="0"
                step="0.01"
                value={form.sellingPrice}
                onChange={(e) => setForm({ ...form, sellingPrice: e.target.value })}
                placeholder="0"
                required
              />
            </div>
          </div>
          <DialogFooter>
            <Button type="button" variant="outline" onClick={() => onOpenChange(false)}>
              Cancel
            </Button>
            <Button type="submit">Add Item</Button>
          </DialogFooter>
        </form>
      </DialogContent>
    </Dialog>
  );
};

export default AddStockDialog;
