import { DailySale } from "@/lib/api";
import { Leaf, Printer } from "lucide-react";
import { Button } from "@/components/ui/button";
import { useRef } from "react";

interface InvoiceViewProps {
  sale: DailySale;
  onPrint?: () => void;
}

const InvoiceView = ({ sale, onPrint }: InvoiceViewProps) => {
  const printRef = useRef<HTMLDivElement>(null);

  const handlePrint = () => {
    if (onPrint) {
      onPrint();
    } else {
      window.print();
    }
  };

  const isPaid = sale.dueAmount === 0;
  const isPartial = sale.paidAmount > 0 && sale.dueAmount > 0;

  return (
    <div className="space-y-4">
      {/* Print Button */}
      <div className="flex justify-end no-print">
        <Button onClick={handlePrint} className="gap-2">
          <Printer className="h-4 w-4" />
          Print Invoice
        </Button>
      </div>

      {/* Invoice Content */}
      <div ref={printRef} className="border border-border rounded-xl p-6 bg-card max-w-4xl mx-auto print:border-0">
        {/* Invoice Header */}
        <div className="flex items-start justify-between mb-6 pb-4 border-b border-border">
          <div className="flex items-center gap-3">
            <div className="bg-primary rounded-lg p-2">
              <Leaf className="h-6 w-6 text-primary-foreground" />
            </div>
            <div>
              <h2 className="text-2xl font-bold font-display text-foreground">Nagar Gram Service</h2>
              <p className="text-sm text-muted-foreground">Professor Para Mor, Noapara, Jessore</p>
            </div>
          </div>
          <div className="text-right">
            <h3 className="text-xl font-bold font-display text-primary">INVOICE</h3>
            <p className="text-sm text-muted-foreground">Invoice No: {sale.invoiceNumber || 'N/A'}</p>
            <p className="text-sm text-muted-foreground">Date: {sale.date}</p>
          </div>
        </div>

        {/* Customer Info */}
        <div className="grid grid-cols-2 gap-6 mb-6">
          <div>
            <p className="text-xs text-muted-foreground font-semibold mb-2">Bill To:</p>
            {sale.customerName ? (
              <>
                <p className="text-foreground font-medium">{sale.customerName}</p>
                {sale.customerAddress && (
                  <p className="text-muted-foreground text-sm">{sale.customerAddress}</p>
                )}
                {sale.customerPhone && (
                  <p className="text-muted-foreground text-sm">Mobile: {sale.customerPhone}</p>
                )}
              </>
            ) : (
              <p className="text-muted-foreground text-sm italic">Walk-in Customer</p>
            )}
          </div>
          <div className="text-right">
            <p className="text-xs text-muted-foreground font-semibold mb-2">Contact:</p>
            <p className="text-foreground text-sm">1855888248</p>
            <p className="text-muted-foreground text-sm">nagargramservice@gmail.com</p>
            <p className="text-muted-foreground text-sm">www.nagargramservice.com</p>
          </div>
        </div>

        {/* Items Table */}
        <div className="border border-border rounded-lg overflow-hidden mb-6">
          <table className="w-full">
            <thead>
              <tr className="bg-muted/50">
                <th className="text-left p-3 text-sm font-semibold text-foreground">SL</th>
                <th className="text-left p-3 text-sm font-semibold text-foreground">Item Name</th>
                <th className="text-right p-3 text-sm font-semibold text-foreground">Unit Price</th>
                <th className="text-right p-3 text-sm font-semibold text-foreground">Qty</th>
                <th className="text-right p-3 text-sm font-semibold text-foreground">Amount</th>
              </tr>
            </thead>
            <tbody>
              {sale.items.map((item, index) => (
                <tr key={index} className="border-t border-border">
                  <td className="p-3 text-muted-foreground">{index + 1}</td>
                  <td className="p-3 text-foreground">{item.itemName}</td>
                  <td className="p-3 text-right text-muted-foreground">৳{item.tp.toFixed(2)}</td>
                  <td className="p-3 text-right text-muted-foreground">{item.qty}</td>
                  <td className="p-3 text-right font-medium text-foreground">
                    ৳{item.totalTP.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {/* Totals */}
        <div className="flex justify-end mb-6">
          <div className="w-64 space-y-2">
            <div className="flex justify-between text-sm">
              <span className="text-muted-foreground">Sub Total:</span>
              <span className="text-foreground font-medium">
                ৳{sale.totalTP.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
              </span>
            </div>
            {sale.discount > 0 && (
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Discount:</span>
                <span className="text-foreground">
                  ৳{sale.discount.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                </span>
              </div>
            )}
            <div className="flex justify-between border-t border-border pt-2">
              <span className="font-bold text-foreground">Total:</span>
              <span className="font-bold text-primary text-lg">
                ৳{sale.totalTP.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
              </span>
            </div>
            {sale.paidAmount > 0 && (
              <>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">Paid Amount:</span>
                  <span className="text-success font-medium">
                    ৳{sale.paidAmount.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                  </span>
                </div>
                {sale.dueAmount > 0 && (
                  <div className="flex justify-between text-sm">
                    <span className="text-muted-foreground">Due Amount:</span>
                    <span className="text-destructive font-medium">
                      ৳{sale.dueAmount.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                    </span>
                  </div>
                )}
              </>
            )}
          </div>
        </div>

        {/* Footer */}
        <div className="text-center border-t border-border pt-4">
          <p className="text-sm text-muted-foreground italic mb-3">
            "Thank you for your purchase. Your support is greatly appreciated."
          </p>
          <div className="inline-block">
            {isPaid && (
              <div className="bg-success/10 text-success text-sm font-bold px-6 py-2 rounded-full">
                PAID
              </div>
            )}
            {isPartial && (
              <div className="bg-warning/10 text-warning text-sm font-bold px-6 py-2 rounded-full">
                PARTIALLY PAID
              </div>
            )}
            {!isPaid && !isPartial && (
              <div className="bg-destructive/10 text-destructive text-sm font-bold px-6 py-2 rounded-full">
                UNPAID
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Print Styles */}
      <style>{`
        @media print {
          .no-print {
            display: none !important;
          }
        }
      `}</style>
    </div>
  );
};

export default InvoiceView;
