import { useState, useCallback } from "react";
import { products } from "@/data/stockData";

export interface Category {
  id: string;
  name: string;
}

// Extract unique categories from existing product data
const extractInitialCategories = (): Category[] => {
  const unique = [...new Set(products.map((p) => p.category))];
  return unique.map((name, i) => ({ id: `cat-${i + 1}`, name }));
};

export const useCategoryManager = () => {
  const [categories, setCategories] = useState<Category[]>(() => extractInitialCategories());

  const addCategory = useCallback((name: string) => {
    const id = `cat-${Date.now()}`;
    setCategories((prev) => [...prev, { id, name }]);
  }, []);

  const updateCategory = useCallback((id: string, name: string) => {
    setCategories((prev) =>
      prev.map((cat) => (cat.id === id ? { ...cat, name } : cat))
    );
  }, []);

  const deleteCategory = useCallback((id: string) => {
    setCategories((prev) => prev.filter((cat) => cat.id !== id));
  }, []);

  return { categories, addCategory, updateCategory, deleteCategory };
};
