import { useState, useCallback, useMemo } from "react";
import { StockItem, stockItems as initialStockItems } from "@/data/stockData";

export interface StockManagerActions {
  addItem: (item: StockItem) => void;
  updateItem: (index: number, item: StockItem) => void;
  deleteItem: (index: number) => void;
  stockIn: (index: number, qty: number, date: string) => void;
  stockOut: (index: number, qty: number, date: string) => void;
}

export const useStockManager = () => {
  const [items, setItems] = useState<StockItem[]>(() => [...initialStockItems]);

  const addItem = useCallback((item: StockItem) => {
    setItems((prev) => [...prev, item]);
  }, []);

  const updateItem = useCallback((index: number, item: StockItem) => {
    setItems((prev) => prev.map((existing, i) => (i === index ? item : existing)));
  }, []);

  const deleteItem = useCallback((index: number) => {
    setItems((prev) => prev.filter((_, i) => i !== index));
  }, []);

  const stockIn = useCallback((index: number, qty: number, date: string) => {
    setItems((prev) =>
      prev.map((item, i) => {
        if (i !== index) return item;
        const newStockIn = item.stockInQty + qty;
        const newBalance = item.stockBalance + qty;
        return {
          ...item,
          stockInQty: newStockIn,
          stockInDate: date,
          stockBalance: newBalance,
          totalStockRate: newBalance * item.costPrice,
        };
      })
    );
  }, []);

  const stockOut = useCallback((index: number, qty: number, date: string) => {
    setItems((prev) =>
      prev.map((item, i) => {
        if (i !== index) return item;
        const actualQty = Math.min(qty, item.stockBalance);
        const newOut = item.stockOutQty + actualQty;
        const newBalance = item.stockBalance - actualQty;
        return {
          ...item,
          stockOutQty: newOut,
          stockOutDate: date,
          stockBalance: newBalance,
          totalSalesRate: item.totalSalesRate + actualQty * item.sellingPrice,
          totalStockRate: newBalance * item.costPrice,
        };
      })
    );
  }, []);

  const stats = useMemo(() => {
    const totalProducts = items.length;
    const totalBalance = items.reduce((sum, item) => sum + item.stockBalance, 0);
    const stockValue = items.reduce((sum, item) => sum + item.totalStockRate, 0);
    const salesValue = items.reduce((sum, item) => sum + item.totalSalesRate, 0);
    const outOfStock = items.filter((item) => item.stockBalance === 0).length;
    const lowStock = items.filter((item) => item.stockBalance > 0 && item.stockBalance <= 10).length;
    return { totalProducts, totalBalance, stockValue, salesValue, outOfStock, lowStock };
  }, [items]);

  return {
    items,
    stats,
    actions: { addItem, updateItem, deleteItem, stockIn, stockOut },
  };
};
