import apiClient from './axios';

// Helper to convert dd/MM/yyyy to yyyy-MM-dd
const convertDateToISO = (dateStr: string): string => {
  const parts = dateStr.split('/');
  if (parts.length === 3) {
    return `${parts[2]}-${parts[1].padStart(2, '0')}-${parts[0].padStart(2, '0')}`;
  }
  return dateStr;
};

// Types
export interface User {
  id: number;
  name: string;
  email: string;
}

export interface Category {
  id: number;
  name: string;
}

export interface Product {
  id: number;
  name: string;
  nickname: string;
  categoryId: number;
  category: string;
}

export interface StockItem {
  id: number;
  itemName: string;
  vendor?: string;
  stockInDate: string;
  stockInQty: number;
  stockOutDate?: string;
  stockOutQty: number;
  costPrice: number;
  stockBalance: number;
  sellingPrice: number;
  totalSalesRate: number;
  totalStockRate: number;
}

export interface DailySaleItem {
  itemName: string;
  qty: number;
  dp: number;
  tp: number;
  totalTP: number;
  totalDP: number;
  profit: number;
}

export interface DailySale {
  id?: number;
  date: string;
  invoiceNumber?: string;
  customerName?: string;
  customerPhone?: string;
  customerAddress?: string;
  items: DailySaleItem[];
  totalTP: number;
  totalDP: number;
  totalProfit: number;
  dueAmount: number;
  paidAmount: number;
  discount: number;
  netProfit: number;
}

// Auth API
export const authApi = {
  login: (email: string, password: string) =>
    apiClient.post('/login', { email, password }),

  register: (name: string, email: string, password: string, password_confirmation: string) =>
    apiClient.post('/register', { name, email, password, password_confirmation }),

  logout: () => apiClient.post('/logout'),

  getUser: () => apiClient.get('/user'),
};

// Categories API
export const categoriesApi = {
  getAll: () => apiClient.get<{ data: Category[] }>('/categories'),

  create: (data: { name: string }) =>
    apiClient.post<{ data: Category }>('/categories', data),

  update: (id: number, data: { name: string }) =>
    apiClient.put<{ data: Category }>(`/categories/${id}`, data),

  delete: (id: number) => apiClient.delete(`/categories/${id}`),
};

// Products API
export const productsApi = {
  getAll: () => apiClient.get<{ data: Product[] }>('/products'),

  create: (data: { name: string; nickname: string; categoryId: number }) =>
    apiClient.post<{ data: Product }>('/products', {
      name: data.name,
      nickname: data.nickname,
      category_id: data.categoryId,
    }),

  update: (id: number, data: { name: string; nickname: string; categoryId: number }) =>
    apiClient.put<{ data: Product }>(`/products/${id}`, {
      name: data.name,
      nickname: data.nickname,
      category_id: data.categoryId,
    }),

  delete: (id: number) => apiClient.delete(`/products/${id}`),
};

// Stock Items API
export const stockApi = {
  getAll: () => apiClient.get<{ data: StockItem[] }>('/stock-items'),

  create: (data: any) =>
    apiClient.post<{ data: StockItem }>('/stock-items', {
      item_name: data.itemName,
      vendor: data.vendor,
      stock_in_date: convertDateToISO(data.stockInDate),
      stock_in_qty: data.stockInQty,
      cost_price: data.costPrice,
      selling_price: data.sellingPrice,
    }),

  update: (id: number, data: any) =>
    apiClient.put<{ data: StockItem }>(`/stock-items/${id}`, {
      item_name: data.itemName,
      vendor: data.vendor,
      stock_in_date: convertDateToISO(data.stockInDate),
      stock_in_qty: data.stockInQty,
      cost_price: data.costPrice,
      selling_price: data.sellingPrice,
    }),

  delete: (id: number) => apiClient.delete(`/stock-items/${id}`),

  stockIn: (id: number, qty: number, date: string) =>
    apiClient.post<{ data: StockItem }>(`/stock-items/${id}/stock-in`, {
      qty,
      date: convertDateToISO(date)
    }),

  stockOut: (id: number, qty: number, date: string) =>
    apiClient.post<{ data: StockItem }>(`/stock-items/${id}/stock-out`, {
      qty,
      date: convertDateToISO(date)
    }),

  getAlerts: () => apiClient.get('/stock-items/alerts'),
};

// Daily Sales API
export const salesApi = {
  getAll: (params?: { start_date?: string; end_date?: string }) =>
    apiClient.get<{ data: DailySale[] }>('/daily-sales', { params }),

  create: (data: DailySale) =>
    apiClient.post<{ data: DailySale }>('/daily-sales', {
      date: convertDateToISO(data.date),
      customer_name: data.customerName || null,
      customer_phone: data.customerPhone || null,
      customer_address: data.customerAddress || null,
      total_tp: data.totalTP,
      total_dp: data.totalDP,
      total_profit: data.totalProfit,
      due_amount: data.dueAmount,
      paid_amount: data.paidAmount,
      discount: data.discount,
      net_profit: data.netProfit,
      items: data.items.map(item => ({
        item_name: item.itemName,
        qty: item.qty,
        dp: item.dp,
        tp: item.tp,
        total_tp: item.totalTP,
        total_dp: item.totalDP,
        profit: item.profit,
      })),
    }),

  delete: (id: number) => apiClient.delete(`/daily-sales/${id}`),
};

// Dashboard API
export const dashboardApi = {
  getStats: () => apiClient.get('/dashboard/stats'),

  getSalesTrend: () => apiClient.get('/dashboard/sales-trend'),

  getTopProducts: () => apiClient.get('/dashboard/top-products'),

  getCategoryDistribution: () => apiClient.get('/dashboard/category-distribution'),
};
