<?php

namespace App\Services;

use App\Models\Product;
use App\Models\StockItem;
use App\Models\DailySale;
use App\Models\DailySaleItem;
use Illuminate\Support\Facades\DB;

class DashboardService
{
    /**
     * Get overall dashboard statistics
     */
    public function getStats(): array
    {
        return [
            'total_products' => Product::count(),
            'total_stock_balance' => StockItem::sum('stock_balance'),
            'stock_value' => StockItem::sum('total_stock_rate'),
            'sales_value' => StockItem::sum('total_sales_rate'),
            'total_profit' => DailySale::sum('total_profit'),
            'out_of_stock_count' => StockItem::outOfStock()->count(),
            'low_stock_count' => StockItem::lowStock()->count(),
        ];
    }

    /**
     * Get sales trend for charts
     */
    public function getSalesTrend(int $days = 30): array
    {
        return DailySale::orderBy('date', 'desc')
            ->take($days)
            ->get()
            ->reverse()
            ->map(fn($sale) => [
                'date' => $sale->date->format('d/m'),
                'sales' => (float) $sale->total_tp,
                'cost' => (float) $sale->total_dp,
                'profit' => (float) $sale->total_profit,
            ])
            ->values()
            ->toArray();
    }

    /**
     * Get top selling products
     */
    public function getTopProducts(int $limit = 8): array
    {
        return DailySaleItem::select('item_name', DB::raw('SUM(qty) as total_qty'))
            ->groupBy('item_name')
            ->orderByDesc('total_qty')
            ->limit($limit)
            ->get()
            ->map(fn($item) => [
                'name' => $item->item_name,
                'qty' => (int) $item->total_qty,
            ])
            ->toArray();
    }

    /**
     * Get category distribution
     */
    public function getCategoryDistribution(): array
    {
        return Product::select('categories.name as category_name', DB::raw('COUNT(*) as count'))
            ->join('categories', 'products.category_id', '=', 'categories.id')
            ->groupBy('categories.name')
            ->get()
            ->map(fn($item) => [
                'name' => $item->category_name,
                'value' => (int) $item->count,
            ])
            ->toArray();
    }
}
