<?php

namespace App\Services;

use App\Models\DailySale;
use App\Models\DailySaleItem;
use App\Models\StockItem;
use Illuminate\Support\Facades\DB;

class SalesService
{
    public function __construct(private StockService $stockService)
    {
    }

    /**
     * Create daily sale and automatically stock out items
     */
    public function createDailySale(array $data): DailySale
    {
        return DB::transaction(function () use ($data) {
            // Find or create daily sale for the date
            $dailySale = DailySale::firstOrNew(['date' => $data['date']]);

            if ($dailySale->exists) {
                // Merge with existing sale
                $dailySale->total_tp += $data['total_tp'];
                $dailySale->total_dp += $data['total_dp'];
                $dailySale->total_profit += $data['total_profit'];
                $dailySale->paid_amount += $data['paid_amount'];
                $dailySale->due_amount += $data['due_amount'];
                $dailySale->discount += $data['discount'];
                $dailySale->net_profit += $data['net_profit'];
            } else {
                $dailySale->fill($data);

                // Generate invoice number if not provided
                if (empty($dailySale->invoice_number)) {
                    $dailySale->invoice_number = $this->generateInvoiceNumber($data['date']);
                }
            }

            $dailySale->save();

            // Create sale items and stock out
            foreach ($data['items'] as $itemData) {
                // Create sale item
                DailySaleItem::create([
                    'daily_sale_id' => $dailySale->id,
                    'item_name' => $itemData['item_name'],
                    'qty' => $itemData['qty'],
                    'dp' => $itemData['dp'],
                    'tp' => $itemData['tp'],
                    'total_tp' => $itemData['total_tp'],
                    'total_dp' => $itemData['total_dp'],
                    'profit' => $itemData['profit'],
                ]);

                // Stock out from inventory
                $stockItem = StockItem::where('item_name', $itemData['item_name'])->first();
                if ($stockItem) {
                    $this->stockService->stockOut($stockItem, $itemData['qty'], $data['date']);
                }
            }

            return $dailySale->load('items');
        });
    }

    /**
     * Generate unique invoice number based on date
     */
    private function generateInvoiceNumber(string $date): string
    {
        $dateObj = is_string($date) ? new \DateTime($date) : $date;
        $dateStr = $dateObj->format('Ymd');

        // Get count of sales for this date
        $count = DailySale::whereDate('date', $dateObj->format('Y-m-d'))->count();
        $sequence = str_pad($count + 1, 3, '0', STR_PAD_LEFT);

        return "INV-{$dateStr}-{$sequence}";
    }
}
